import 'package:flutter/material.dart';
import 'package:flutter_spinkit/flutter_spinkit.dart';
import 'package:get/get.dart';
import 'package:viserpay/core/helper/date_converter.dart';
import 'package:viserpay/core/helper/string_format_helper.dart';
import 'package:viserpay/core/utils/dimensions.dart';
import 'package:viserpay/core/utils/my_color.dart';
import 'package:viserpay/core/utils/my_strings.dart';
import 'package:viserpay/core/utils/status.dart';
import 'package:viserpay/core/utils/style.dart';
import 'package:viserpay/core/utils/url_container.dart';
import 'package:viserpay/data/controller/savings/savings_history_controller.dart';
import 'package:viserpay/data/repo/saving/saving_repo.dart';
import 'package:viserpay/data/services/api_service.dart';
import 'package:viserpay/view/components/app-bar/custom_appbar.dart';
import 'package:viserpay/view/components/bottom-sheet/custom_bottom_sheet.dart';
import 'package:viserpay/view/components/buttons/gradient_rounded_button.dart';
import 'package:viserpay/view/components/buttons/gradient_rounded_danger_button.dart';
import 'package:viserpay/view/components/column_widget/card_column.dart';
import 'package:viserpay/view/components/custom_loader/custom_loader.dart';
import 'package:viserpay/view/components/dialog/app_dialog.dart';
import 'package:viserpay/view/components/divider/custom_divider.dart';
import 'package:viserpay/view/screens/savings/widget/close_account_bottom_sheet.dart';
import 'package:viserpay/view/screens/savings/widget/dps_history_widget.dart';

class InstallmentScreen extends StatefulWidget {
  const InstallmentScreen({super.key});

  @override
  State<InstallmentScreen> createState() => _InstallmentScreenState();
}

class _InstallmentScreenState extends State<InstallmentScreen> {
  @override
  void initState() {
    Get.put(ApiClient(sharedPreferences: Get.find()));
    Get.put(SavingRepo(apiClient: Get.find()));
    final controller = Get.put(SavingsHistoryController(repo: Get.find()));

    super.initState();
    WidgetsBinding.instance.addPostFrameCallback((t) {
      controller.getDpsDetails();
    });
  }

  @override
  Widget build(BuildContext context) {
    return Scaffold(
      backgroundColor: MyColor.screenBgColor,
      appBar: CustomAppBar(title: MyStrings.installment, isTitleCenter: true),
      body: GetBuilder<SavingsHistoryController>(
        builder: (controller) {
          return RefreshIndicator(
            onRefresh: () async {
              controller.getDpsDetails();
            },
            child: SingleChildScrollView(
              padding: Dimensions.defaultPaddingHV,
              child: controller.isDetailsLoading
                  ? CustomLoader(isFullScreen: true)
                  : Column(
                      children: [
                        if (controller.dps != null) ...[
                          DpsHistoryWidget(
                            dps: controller.dps!,
                            currency: controller.repo.apiClient.getCurrencyOrUsername(isSymbol: true),
                            imagePath: "${UrlContainer.domainUrl}/${controller.organizationImagePath}/${controller.dps?.dpsPlan?.organization?.image}",
                            showStatus: controller.dps?.status != MyStatus.ACCOUNT_ACTIVE ? true : false,
                            onTap: () {
                              //
                            },
                          ),
                          SizedBox(height: 10),
                          Container(
                            padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: Dimensions.space10),
                            decoration: BoxDecoration(
                              color: MyColor.colorWhite,
                              borderRadius: BorderRadius.circular(Dimensions.mediumRadius),
                            ),
                            child: Row(
                              mainAxisAlignment: MainAxisAlignment.spaceBetween,
                              children: [
                                CardColumn(
                                  header: "Balance",
                                  body: "${controller.repo.apiClient.getCurrencyOrUsername(isCurrency: true, isSymbol: true)}${StringConverter.formatNumber(controller.dps?.balance ?? "", precision: 2)}",
                                  headerTextStyle: boldDefault.copyWith(fontSize: Dimensions.fontLarge - 1, fontWeight: FontWeight.w500),
                                  bodyTextStyle: lightMediumLarge.copyWith(color: MyColor.getGreyText()),
                                ),
                                CardColumn(
                                  header: MyStrings.interest,
                                  body: "${StringConverter.formatNumber(controller.dps?.dpsPlan?.interestRate ?? "", precision: 2)}%",
                                  headerTextStyle: boldDefault.copyWith(fontSize: Dimensions.fontLarge - 1, fontWeight: FontWeight.w500),
                                  bodyTextStyle: lightMediumLarge.copyWith(color: MyColor.getGreyText()),
                                ),
                                CardColumn(
                                  header: MyStrings.tenure,
                                  body: controller.dps?.dpsPlan?.tenure?.name ?? "",
                                  headerTextStyle: boldDefault.copyWith(fontSize: Dimensions.fontLarge - 1, fontWeight: FontWeight.w500),
                                  bodyTextStyle: lightMediumLarge.copyWith(color: MyColor.getGreyText()),
                                  alignmentEnd: true,
                                ),
                              ],
                            ),
                          ),
                          SizedBox(height: Dimensions.space10),
                          if (controller.dps?.status == MyStatus.ACCOUNT_ACTIVE) ...[
                            Container(
                              padding: EdgeInsets.symmetric(horizontal: Dimensions.space15, vertical: Dimensions.space15),
                              width: double.infinity,
                              decoration: BoxDecoration(
                                color: MyColor.pendingColor.withValues(alpha: 0.1),
                                border: Border.all(color: MyColor.pendingColor, width: .5),
                                borderRadius: BorderRadius.circular(Dimensions.largeRadius),
                              ),
                              child: Stack(
                                children: [
                                  Column(
                                    crossAxisAlignment: CrossAxisAlignment.start,
                                    children: [
                                      Text("Next Installment Date", style: regularDefault.copyWith(color: MyColor.pendingColor)),
                                      SizedBox(height: Dimensions.space5),
                                      Text(DateConverter.convertIsoToStringDPS(controller.dps?.nextInstallmentDate ?? ''), style: semiBoldDefault.copyWith(color: MyColor.colorBlack), textAlign: TextAlign.center),
                                      SizedBox(height: Dimensions.space5),
                                      Text("Missed installment? a ${StringConverter.formatNumber(controller.dps?.dpsPlan?.penaltyRate ?? '', precision: 0)}% penalty applies after this due date", style: regularDefault.copyWith(color: MyColor.colorRed)),
                                    ],
                                  ),
                                  Positioned(
                                    right: 0,
                                    top: 0,
                                    child: Icon(
                                      Icons.calendar_month,
                                      color: MyColor.colorGrey,
                                    ),
                                  )
                                ],
                              ),
                            )
                          ]
                        ],
                        CustomDivider(color: MyColor.primaryColor),
                        ListView.separated(
                          separatorBuilder: (context, index) => SizedBox(height: 15),
                          itemCount: controller.installment.length,
                          physics: NeverScrollableScrollPhysics(),
                          shrinkWrap: true,
                          itemBuilder: (context, index) {
                            final ins = controller.installment[index];
                            return Container(
                              padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: Dimensions.space10),
                              decoration: BoxDecoration(
                                color: MyColor.colorWhite,
                                borderRadius: BorderRadius.circular(Dimensions.mediumRadius),
                              ),
                              child: Column(
                                children: [
                                  SizedBox(height: Dimensions.space10),
                                  customRow(title: MyStrings.installmentNo, value: "#${ins.trx}"),
                                  customRow(title: MyStrings.installmentAmount, value: "${controller.repo.apiClient.getCurrencyOrUsername(isCurrency: true, isSymbol: true)}${StringConverter.formatNumber(ins.amount ?? '0.0')}", valueTextStyle: regularDefault.copyWith(color: MyColor.greenSuccessColor)),
                                  if (StringConverter.formatNumber(ins.penaltyAmount ?? "0.00") != "0.00") ...[
                                    customRow(title: MyStrings.penaltyAmount, value: "${controller.repo.apiClient.getCurrencyOrUsername(isCurrency: true, isSymbol: true)}${StringConverter.formatNumber(ins.penaltyAmount ?? "")}", valueTextStyle: regularDefault.copyWith(color: MyColor.redCancelTextColor)),
                                  ],
                                  customRow(title: MyStrings.installmentDate, value: DateConverter.convertIsoToString(ins.createdAt ?? "")),
                                  installmentStatus(ins.status.toString()),
                                  SizedBox(height: Dimensions.space10),
                                ],
                              ),
                            );
                          },
                        )
                      ],
                    ),
            ),
          );
        },
      ),
      bottomNavigationBar: GetBuilder<SavingsHistoryController>(builder: (controller) {
        return controller.isDetailsLoading
            ? SizedBox.shrink()
            : Container(
                margin: EdgeInsets.all(10),
                constraints: BoxConstraints.expand(height: 60),
                child: controller.dps?.status.toString() != MyStatus.ACCOUNT_ACTIVE
                    ? SizedBox.shrink()
                    : InkWell(
                        onTap: () {
                          CustomBottomSheet(child: CloseAccountBottomSheet()).customBottomSheet(context);
                        },
                        customBorder: RoundedRectangleBorder(borderRadius: BorderRadius.circular(Dimensions.largeRadius)),
                        child: Container(
                          padding: EdgeInsets.symmetric(horizontal: Dimensions.space15, vertical: Dimensions.space15),
                          width: double.infinity,
                          decoration: BoxDecoration(
                            color: MyColor.redCancelTextColor,
                            border: Border.all(color: MyColor.redCancelTextColor, width: .5),
                            borderRadius: BorderRadius.circular(Dimensions.largeRadius),
                          ),
                          child: controller.isCloseBtnLoading
                              ? SizedBox(
                                  height: 20,
                                  width: 20,
                                  child: SpinKitCircle(color: MyColor.colorWhite, size: 20.0),
                                )
                              : Center(child: Text("Close Account", style: regularDefault.copyWith(color: MyColor.colorWhite))),
                        ),
                      ),
              );
      }),
    );
  }

  Widget accountStatusWidget(String status) {
    return Container(
      padding: EdgeInsets.symmetric(horizontal: Dimensions.space15, vertical: Dimensions.space15),
      width: double.infinity,
      decoration: BoxDecoration(
        color: status == MyStatus.ACCOUNT_CLOSED ? MyColor.redCancelTextColor.withValues(alpha: 0.1) : MyColor.greenSuccessColor.withValues(alpha: 0.1),
        border: Border.all(color: status == MyStatus.ACCOUNT_CLOSED ? MyColor.redCancelTextColor : MyColor.greenSuccessColor, width: .5),
        borderRadius: BorderRadius.circular(Dimensions.largeRadius),
      ),
      child: Column(
        children: [
          Text(
            status == MyStatus.ACCOUNT_CLOSED ? "Your account is closed" : "Your account is matured",
            style: regularDefault.copyWith(color: status == MyStatus.ACCOUNT_CLOSED ? MyColor.redCancelTextColor : MyColor.greenSuccessColor),
          ),
          SizedBox(height: Dimensions.space5),
          Text(
            status == MyStatus.ACCOUNT_CLOSED ? "You cannot get the interest amount" : "Your matured amount already added with your account balance",
            style: lightDefault.copyWith(color: MyColor.colorGrey.withValues(alpha: 0.5)),
            textAlign: TextAlign.center,
          ),
        ],
      ),
    );
  }

  Widget installmentStatus(String status) {
    return Row(
      mainAxisAlignment: MainAxisAlignment.spaceBetween,
      crossAxisAlignment: CrossAxisAlignment.center,
      children: [
        Text(MyStrings.status.tr, style: semiBoldDefault.copyWith(fontSize: Dimensions.fontLarge - 1)),
        status == MyStatus.INSTALLMENT_PAID
            ? Container(
                padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: Dimensions.space5),
                decoration: BoxDecoration(
                  color: MyColor.colorGreen.withValues(alpha: 0.1),
                  border: Border.all(color: MyColor.colorGreen, width: .5),
                  borderRadius: BorderRadius.circular(Dimensions.mediumRadius),
                ),
                child: Text(MyStrings.paid.tr, style: lightDefault.copyWith(fontSize: Dimensions.fontLarge - 1, color: MyColor.colorGreen)),
              )
            : Container(
                padding: EdgeInsets.symmetric(horizontal: Dimensions.space10, vertical: Dimensions.space5),
                decoration: BoxDecoration(
                  color: MyColor.pendingColor.withValues(alpha: 0.1),
                  border: Border.all(color: MyColor.pendingColor, width: .5),
                  borderRadius: BorderRadius.circular(Dimensions.mediumRadius),
                ),
                child: Text(MyStrings.unpaid.tr, style: lightDefault.copyWith(fontSize: Dimensions.fontLarge - 1, color: MyColor.pendingColor)),
              ),
      ],
    );
  }

  Widget customRow({required String title, required String value, bool showDivider = true, TextStyle? valueTextStyle}) {
    return Column(
      children: [
        Row(
          mainAxisAlignment: MainAxisAlignment.spaceBetween,
          crossAxisAlignment: CrossAxisAlignment.center,
          children: [
            Expanded(child: Text(title.tr, style: semiBoldDefault.copyWith(fontSize: Dimensions.fontLarge - 1))),
            Expanded(
              child: Text(
                value.tr,
                style: valueTextStyle ?? lightDefault.copyWith(fontSize: Dimensions.fontLarge - 1, color: MyColor.bodyTextColor),
                textAlign: TextAlign.end,
                maxLines: 2,
                overflow: TextOverflow.ellipsis,
              ),
            ),
          ],
        ),
        if (showDivider) CustomDivider(onlyBottom: true),
      ],
    );
  }
}
